/*
 * Copyright 2019-2025 The OpenSSL Project Authors. All Rights Reserved.
 *
 * Licensed under the Apache License 2.0 (the "License").  You may not use
 * this file except in compliance with the License.  You can obtain a copy
 * in the file LICENSE in the source distribution or at
 * https://www.openssl.org/source/license.html
 */


#include <string.h>
#include <openssl/crypto.h>
#include <openssl/core_names.h>
#include <openssl/proverr.h>
#include <openssl/err.h>
#include "internal/cryptlib.h"
#include "prov/blake2.h"
#include "prov/digestcommon.h"
#include "prov/implementations.h"

static OSSL_FUNC_digest_gettable_ctx_params_fn blake_gettable_ctx_params;
static OSSL_FUNC_digest_settable_ctx_params_fn blake_settable_ctx_params;

/* Machine generated by util/perl/OpenSSL/paramnames.pm */
#ifndef blake_get_ctx_params_list
static const OSSL_PARAM blake_get_ctx_params_list[] = {
    OSSL_PARAM_uint(OSSL_DIGEST_PARAM_SIZE, NULL),
    OSSL_PARAM_END
};
#endif

#ifndef blake_get_ctx_params_st
struct blake_get_ctx_params_st {
    OSSL_PARAM *size;
};
#endif

#ifndef blake_get_ctx_params_decoder
static int blake_get_ctx_params_decoder
    (const OSSL_PARAM *p, struct blake_get_ctx_params_st *r)
{
    const char *s;

    memset(r, 0, sizeof(*r));
    if (p != NULL)
        for (; (s = p->key) != NULL; p++)
            if (ossl_likely(strcmp("size", s + 0) == 0)) {
                /* OSSL_DIGEST_PARAM_SIZE */
                if (ossl_unlikely(r->size != NULL)) {
                    ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                   "param %s is repeated", s);
                    return 0;
                }
                r->size = (OSSL_PARAM *)p;
            }
    return 1;
}
#endif
/* End of machine generated */

static const OSSL_PARAM *blake_gettable_ctx_params(ossl_unused void *ctx,
                                                   ossl_unused void *pctx)
{
    return blake_get_ctx_params_list;
}

/* Machine generated by util/perl/OpenSSL/paramnames.pm */
#ifndef blake_set_ctx_params_list
static const OSSL_PARAM blake_set_ctx_params_list[] = {
    OSSL_PARAM_uint(OSSL_DIGEST_PARAM_SIZE, NULL),
    OSSL_PARAM_END
};
#endif

#ifndef blake_set_ctx_params_st
struct blake_set_ctx_params_st {
    OSSL_PARAM *size;
};
#endif

#ifndef blake_set_ctx_params_decoder
static int blake_set_ctx_params_decoder
    (const OSSL_PARAM *p, struct blake_set_ctx_params_st *r)
{
    const char *s;

    memset(r, 0, sizeof(*r));
    if (p != NULL)
        for (; (s = p->key) != NULL; p++)
            if (ossl_likely(strcmp("size", s + 0) == 0)) {
                /* OSSL_DIGEST_PARAM_SIZE */
                if (ossl_unlikely(r->size != NULL)) {
                    ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                   "param %s is repeated", s);
                    return 0;
                }
                r->size = (OSSL_PARAM *)p;
            }
    return 1;
}
#endif
/* End of machine generated */

static const OSSL_PARAM *blake_settable_ctx_params(ossl_unused void *ctx,
                                                   ossl_unused void *pctx)
{
    return blake_set_ctx_params_list;
}

#define IMPLEMENT_BLAKE_functions(variant, VARIANT, variantsize) \
int ossl_blake##variant##_get_ctx_params(void *vctx, OSSL_PARAM params[]) \
{ \
    struct blake##variant##_md_data_st *mdctx = vctx; \
    struct blake_get_ctx_params_st p; \
 \
    BLAKE##VARIANT##_CTX *ctx = &mdctx->ctx; \
 \
    if (ctx == NULL || !blake_get_ctx_params_decoder(params, &p)) \
        return 0; \
 \
    if (p.size != NULL \
        && !OSSL_PARAM_set_uint(p.size, (unsigned int)mdctx->params.digest_length)) { \
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER); \
        return 0; \
    } \
 \
    return 1; \
} \
 \
int ossl_blake##variant##_set_ctx_params(void *vctx, const OSSL_PARAM params[]) \
{ \
    unsigned int size; \
    struct blake##variant##_md_data_st *mdctx = vctx; \
    struct blake_set_ctx_params_st p; \
 \
    BLAKE##VARIANT##_CTX *ctx = &mdctx->ctx; \
 \
    if (ctx == NULL || !blake_set_ctx_params_decoder(params, &p)) \
        return 0; \
 \
    if (p.size != NULL) { \
        if (!OSSL_PARAM_get_uint(p.size, &size)) { \
            ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_GET_PARAMETER); \
            return 0; \
        } \
        if (size < 1 || size > BLAKE##VARIANT##_OUTBYTES) { \
            ERR_raise(ERR_LIB_PROV, PROV_R_INVALID_DIGEST_SIZE); \
            return 0; \
        } \
        ossl_blake##variant##_param_set_digest_length(&mdctx->params, (uint8_t)size); \
    } \
 \
    return 1; \
} \
 \
static int ossl_blake##variantsize##_init(void *ctx) \
{ \
    struct blake##variant##_md_data_st *mdctx = ctx; \
    uint8_t digest_length = mdctx->params.digest_length; \
 \
    ossl_blake##variant##_param_init(&mdctx->params); \
    if (digest_length != 0) \
        mdctx->params.digest_length = digest_length; \
    return ossl_blake##variant##_init(&mdctx->ctx, &mdctx->params); \
} \
 \
static OSSL_FUNC_digest_init_fn blake##variantsize##_internal_init; \
static OSSL_FUNC_digest_newctx_fn blake##variantsize##_newctx; \
static OSSL_FUNC_digest_freectx_fn blake##variantsize##_freectx; \
static OSSL_FUNC_digest_dupctx_fn blake##variantsize##_dupctx; \
static OSSL_FUNC_digest_final_fn blake##variantsize##_internal_final; \
static OSSL_FUNC_digest_get_params_fn blake##variantsize##_get_params; \
 \
static int blake##variantsize##_internal_init(void *ctx, const OSSL_PARAM params[]) \
{ \
    return ossl_prov_is_running() && ossl_blake##variant##_set_ctx_params(ctx, params) \
        && ossl_blake##variantsize##_init(ctx); \
} \
 \
static void *blake##variantsize##_newctx(void *prov_ctx) \
{ \
    struct blake##variant##_md_data_st *ctx; \
 \
    ctx = ossl_prov_is_running() ? OPENSSL_zalloc(sizeof(*ctx)) : NULL; \
    return ctx; \
} \
 \
static void blake##variantsize##_freectx(void *vctx) \
{ \
    struct blake##variant##_md_data_st *ctx; \
 \
    ctx = (struct blake##variant##_md_data_st *)vctx; \
    OPENSSL_clear_free(ctx, sizeof(*ctx)); \
} \
 \
static void *blake##variantsize##_dupctx(void *ctx) \
{ \
    struct blake##variant##_md_data_st *in, *ret; \
 \
    in = (struct blake##variant##_md_data_st *)ctx; \
    ret = ossl_prov_is_running()? OPENSSL_malloc(sizeof(*ret)) : NULL; \
    if (ret != NULL) \
        *ret = *in; \
    return ret; \
} \
\
static void blake##variantsize##_copyctx(void *voutctx, void *vinctx) \
{ \
    struct blake##variant##_md_data_st *inctx, *outctx; \
 \
    outctx = (struct blake##variant##_md_data_st *)voutctx; \
    inctx = (struct blake##variant##_md_data_st *)vinctx; \
    *outctx = *inctx; \
} \
 \
static int blake##variantsize##_internal_final(void *ctx, unsigned char *out, \
                                     size_t *outl, size_t outsz) \
{ \
    struct blake##variant##_md_data_st *b_ctx; \
 \
    b_ctx = (struct blake##variant##_md_data_st *)ctx; \
 \
    if (!ossl_prov_is_running()) \
        return 0; \
 \
    *outl = b_ctx->ctx.outlen; \
 \
    if (outsz == 0) \
       return 1; \
 \
    if (outsz < *outl) { \
        ERR_raise(ERR_LIB_PROV, PROV_R_INVALID_DIGEST_SIZE); \
        return 0; \
    } \
 \
    return ossl_blake##variant##_final(out, ctx); \
} \
 \
static int blake##variantsize##_get_params(OSSL_PARAM params[]) \
{ \
    return ossl_digest_default_get_params(params, BLAKE##VARIANT##_BLOCKBYTES, BLAKE##VARIANT##_OUTBYTES, 0); \
} \
 \
const OSSL_DISPATCH ossl_blake##variantsize##_functions[] = { \
    {OSSL_FUNC_DIGEST_NEWCTX, (void (*)(void))blake##variantsize##_newctx}, \
    {OSSL_FUNC_DIGEST_UPDATE, (void (*)(void))ossl_blake##variant##_update}, \
    {OSSL_FUNC_DIGEST_FINAL, (void (*)(void))blake##variantsize##_internal_final}, \
    {OSSL_FUNC_DIGEST_FREECTX, (void (*)(void))blake##variantsize##_freectx}, \
    {OSSL_FUNC_DIGEST_DUPCTX, (void (*)(void))blake##variantsize##_dupctx}, \
    {OSSL_FUNC_DIGEST_COPYCTX, (void (*)(void))blake##variantsize##_copyctx}, \
    {OSSL_FUNC_DIGEST_GET_PARAMS, (void (*)(void))blake##variantsize##_get_params}, \
    {OSSL_FUNC_DIGEST_GETTABLE_PARAMS, \
     (void (*)(void))ossl_digest_default_gettable_params}, \
    {OSSL_FUNC_DIGEST_INIT, (void (*)(void))blake##variantsize##_internal_init}, \
    {OSSL_FUNC_DIGEST_GETTABLE_CTX_PARAMS, \
     (void (*)(void))blake_gettable_ctx_params}, \
    {OSSL_FUNC_DIGEST_SETTABLE_CTX_PARAMS, \
     (void (*)(void))blake_settable_ctx_params}, \
    {OSSL_FUNC_DIGEST_GET_CTX_PARAMS, \
     (void (*)(void))ossl_blake##variant##_get_ctx_params}, \
    {OSSL_FUNC_DIGEST_SET_CTX_PARAMS, \
     (void (*)(void))ossl_blake##variant##_set_ctx_params}, \
    {0, NULL} \
};

IMPLEMENT_BLAKE_functions(2s, 2S, 2s256)
IMPLEMENT_BLAKE_functions(2b, 2B, 2b512)
