/******************************************************************************
Copyright (c) W.J. van der Laan

Permission is hereby granted, free of charge, to any person obtaining a copy of 
this software  and associated documentation files (the "Software"), to deal in 
the Software without restriction, including without limitation the rights to use, 
copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
Software, and to permit persons to whom the Software is furnished to do so, subject 
to the following conditions:

The above copyright notice and this permission notice shall be included in all copies 
or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A 
PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION 
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,OUT OF OR IN CONNECTION WITH THE 
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
******************************************************************************/

#include "LightMaterialGenerator.h"

#include "OgreStringConverter.h"
#include "OgreException.h"
#include "OgreMaterialManager.h"

#include "OgrePass.h"
#include "OgreTechnique.h"

#include "OgreGpuProgramManager.h"
#include "OgreHighLevelGpuProgram.h"
#include "OgreHighLevelGpuProgramManager.h"

#include "DLight.h"

using namespace Ogre;

//CG
class LightMaterialGeneratorCG : public MaterialGenerator::Impl
{
public:
    typedef MaterialGenerator::Perm Perm;
    LightMaterialGeneratorCG(const String &baseName):
        mBaseName(baseName) 
    {

    }
    virtual ~LightMaterialGeneratorCG()
    {

    }

    GpuProgramPtr generateVertexShader(Perm permutation) override
    {
        String programName = "DeferredShading/post/";

        if (permutation & LightMaterialGenerator::MI_DIRECTIONAL)
        {
            programName += "vs";
        }
        else
        {
            programName += "LightMaterial_vs";
        }

        GpuProgramPtr ptr = HighLevelGpuProgramManager::getSingleton().getByName(
            programName, ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME);
        OgreAssert(ptr, "vertex shader is NULL");
        return ptr;
    }

    GpuProgramPtr generateFragmentShader(Perm permutation) override
    {
        /// Create shader
        if (mMasterSource.empty())
        {
            DataStreamPtr ptrMasterSource = ResourceGroupManager::getSingleton().openResource(
                 "LightMaterial_ps.cg"
                , ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME);
            assert(ptrMasterSource);
            mMasterSource = ptrMasterSource->getAsString();
        }

        assert(mMasterSource.empty()==false);

        // Create name
        String name = mBaseName+StringConverter::toString(permutation)+"_ps";       

        // Create shader object
        HighLevelGpuProgramPtr ptrProgram = HighLevelGpuProgramManager::getSingleton().createProgram(
            name, ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME,
            "hlsl", GPT_FRAGMENT_PROGRAM);
        ptrProgram->setSource(mMasterSource);
        ptrProgram->setParameter("entry_point","main");
        ptrProgram->setParameter("target","ps_2_a");
        // set up the preprocessor defines
        // Important to do this before any call to get parameters, i.e. before the program gets loaded
        ptrProgram->setParameter("preprocessor_defines", getPPDefines(permutation));

        setUpBaseParameters(ptrProgram->getDefaultParameters());

        return GpuProgramPtr(ptrProgram);
    }

    MaterialPtr generateTemplateMaterial(Perm permutation) override
    {
        String materialName = mBaseName;
    
        if(permutation & LightMaterialGenerator::MI_DIRECTIONAL)
        {   
            materialName += "Quad";
        }
        else
        {
            materialName += "Geometry";
        }

        if(permutation & LightMaterialGenerator::MI_SHADOW_CASTER)
        {
            materialName += "Shadow";
        }
        return MaterialManager::getSingleton().getByName(materialName);
    }

    protected:
        String mBaseName;
        String mMasterSource;
        // Utility method
        String getPPDefines(Perm permutation)
        {
            String strPPD;

            //Get the type of light
            String lightType;
            if (permutation & LightMaterialGenerator::MI_POINT)
            {
                lightType = "POINT";
            }
            else if (permutation & LightMaterialGenerator::MI_SPOTLIGHT)
            {
                lightType = "SPOT";
            }
            else if (permutation & LightMaterialGenerator::MI_DIRECTIONAL)
            {
                lightType = "DIRECTIONAL";
            }
            else
            {
                assert(false && "Permutation must have a light type");
            }
            strPPD += "LIGHT_TYPE=LIGHT_" + lightType;

            //Optional parameters
            if (permutation & LightMaterialGenerator::MI_SPECULAR)
            {
                strPPD += ",IS_SPECULAR=1";
            }
            if (permutation & LightMaterialGenerator::MI_ATTENUATED)
            {
                strPPD += ",IS_ATTENUATED=1";
            }
            if (permutation & LightMaterialGenerator::MI_SHADOW_CASTER)
            {
                strPPD += ",IS_SHADOW_CASTER=1";
            }
            return strPPD;
        }

        void setUpBaseParameters(const GpuProgramParametersSharedPtr& params)
        {
            assert(params);

            struct AutoParamPair { String name; GpuProgramParameters::AutoConstantType type; }; 

            //A list of auto params that might be present in the shaders generated
            static const AutoParamPair AUTO_PARAMS[] = {
                { "vpWidth",            GpuProgramParameters::ACT_VIEWPORT_WIDTH },
                { "vpHeight",           GpuProgramParameters::ACT_VIEWPORT_HEIGHT },
                { "worldView",          GpuProgramParameters::ACT_WORLDVIEW_MATRIX },
                { "invProj",            GpuProgramParameters::ACT_INVERSE_PROJECTION_MATRIX },
                { "invView",            GpuProgramParameters::ACT_INVERSE_VIEW_MATRIX },
                { "flip",               GpuProgramParameters::ACT_RENDER_TARGET_FLIPPING },
                { "lightDiffuseColor",  GpuProgramParameters::ACT_LIGHT_DIFFUSE_COLOUR },
                { "lightSpecularColor", GpuProgramParameters::ACT_LIGHT_SPECULAR_COLOUR },
                { "lightFalloff",       GpuProgramParameters::ACT_LIGHT_ATTENUATION },
                { "lightPos",           GpuProgramParameters::ACT_LIGHT_POSITION_VIEW_SPACE },
                { "lightDir",           GpuProgramParameters::ACT_LIGHT_DIRECTION_VIEW_SPACE },
                { "spotParams",         GpuProgramParameters::ACT_SPOTLIGHT_PARAMS },
                { "farClipDistance",    GpuProgramParameters::ACT_FAR_CLIP_DISTANCE },
                { "shadowViewProjMat",  GpuProgramParameters::ACT_TEXTURE_VIEWPROJ_MATRIX }
            };
            int numParams = sizeof(AUTO_PARAMS) / sizeof(AutoParamPair);

            for (int i=0; i<numParams; i++)
            {
                if (params->_findNamedConstantDefinition(AUTO_PARAMS[i].name))
                {
                    params->setNamedAutoConstant(AUTO_PARAMS[i].name, AUTO_PARAMS[i].type);
                }
            }
        }
};

//GLSL
class LightMaterialGeneratorGLSL : public MaterialGenerator::Impl
{
public:
    typedef MaterialGenerator::Perm Perm;
    LightMaterialGeneratorGLSL(const String &baseName):
    mBaseName(baseName)
    {

    }
    virtual ~LightMaterialGeneratorGLSL()
    {

    }

    GpuProgramPtr generateVertexShader(Perm permutation) override
    {
        String programName = "DeferredShading/post/";

        if (permutation & LightMaterialGenerator::MI_DIRECTIONAL)
        {
            programName += "vs";
        }
        else
        {
            programName += "LightMaterial_vs";
        }

        GpuProgramPtr ptr = HighLevelGpuProgramManager::getSingleton().getByName(
            programName, ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME);
        OgreAssert(ptr, "vertex shader is NULL");
        return ptr;
    }

    GpuProgramPtr generateFragmentShader(Perm permutation) override
    {
        /// Create shader
        if (mMasterSource.empty())
        {
            DataStreamPtr ptrMasterSource =
                ResourceGroupManager::getSingleton().openResource("LightMaterial_ps.glsl", RGN_DEFAULT);

            OgreAssert(ptrMasterSource, "could not find 'LightMaterial_ps'");
            mMasterSource = ptrMasterSource->getAsString();
        }

        OgreAssert(!mMasterSource.empty(), "no source code");

        // Create name
        String name = mBaseName+StringConverter::toString(permutation)+"_ps";

        // Create shader object
        HighLevelGpuProgramPtr ptrProgram;
        if(GpuProgramManager::getSingleton().isSyntaxSupported("glsles"))
        {
            ptrProgram = HighLevelGpuProgramManager::getSingleton().createProgram(name, ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME,
                                                                                  "glsles", GPT_FRAGMENT_PROGRAM);
        }
        else
        {
            ptrProgram = HighLevelGpuProgramManager::getSingleton().createProgram(name, ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME,
                                                                                  "glsl", GPT_FRAGMENT_PROGRAM);
        }
        ptrProgram->setSource(mMasterSource);
        // set up the preprocessor defines
        // Important to do this before any call to get parameters, i.e. before the program gets loaded
        ptrProgram->setParameter("preprocessor_defines", getPPDefines(permutation));

        setUpBaseParameters(ptrProgram->getDefaultParameters());

        // Bind samplers
        GpuProgramParametersSharedPtr params = ptrProgram->getDefaultParameters();
        int numSamplers = 0;
        params->setNamedConstant("Tex0", (int)numSamplers++);
        params->setNamedConstant("Tex1", (int)numSamplers++);

        if(permutation & LightMaterialGenerator::MI_SHADOW_CASTER)
            params->setNamedConstant("ShadowTex", (int)numSamplers++);

        return GpuProgramPtr(ptrProgram);
    }

    MaterialPtr generateTemplateMaterial(Perm permutation) override
    {
        String materialName = mBaseName;

        if(permutation & LightMaterialGenerator::MI_DIRECTIONAL)
        {
            materialName += "Quad";
        }
        else
        {
            materialName += "Geometry";
        }

        if(permutation & LightMaterialGenerator::MI_SHADOW_CASTER)
        {
            materialName += "Shadow";
        }
        return MaterialManager::getSingleton().getByName(materialName);
    }

protected:
    String mBaseName;
    String mMasterSource;
    // Utility method
    String getPPDefines(Perm permutation)
    {
        String strPPD;

        //Get the type of light
        Ogre::uint lightType = 0;
        if (permutation & LightMaterialGenerator::MI_POINT)
        {
            lightType = 1;
        }
        else if (permutation & LightMaterialGenerator::MI_SPOTLIGHT)
        {
            lightType = 2;
        }
        else if (permutation & LightMaterialGenerator::MI_DIRECTIONAL)
        {
            lightType = 3;
        }
        else
        {
            assert(false && "Permutation must have a light type");
        }
        strPPD += "LIGHT_TYPE=" + StringConverter::toString(lightType);

        //Optional parameters
        if (permutation & LightMaterialGenerator::MI_SPECULAR)
        {
            strPPD += ",IS_SPECULAR=1";
        }
        if (permutation & LightMaterialGenerator::MI_ATTENUATED)
        {
            strPPD += ",IS_ATTENUATED=1";
        }
        if (permutation & LightMaterialGenerator::MI_SHADOW_CASTER)
        {
            strPPD += ",IS_SHADOW_CASTER=1";
        }
        return strPPD;
    }

    void setUpBaseParameters(const GpuProgramParametersSharedPtr& params)
    {
        assert(params);

        struct AutoParamPair { String name; GpuProgramParameters::AutoConstantType type; };

        //A list of auto params that might be present in the shaders generated
        static const AutoParamPair AUTO_PARAMS[] = {
            { "vpWidth",            GpuProgramParameters::ACT_VIEWPORT_WIDTH },
            { "vpHeight",           GpuProgramParameters::ACT_VIEWPORT_HEIGHT },
            { "worldView",          GpuProgramParameters::ACT_WORLDVIEW_MATRIX },
            { "invProj",            GpuProgramParameters::ACT_INVERSE_PROJECTION_MATRIX },
            { "invView",            GpuProgramParameters::ACT_INVERSE_VIEW_MATRIX },
            { "flip",               GpuProgramParameters::ACT_RENDER_TARGET_FLIPPING },
            { "lightDiffuseColor",  GpuProgramParameters::ACT_LIGHT_DIFFUSE_COLOUR },
            { "lightSpecularColor", GpuProgramParameters::ACT_LIGHT_SPECULAR_COLOUR },
            { "lightFalloff",       GpuProgramParameters::ACT_LIGHT_ATTENUATION },
            { "lightPos",           GpuProgramParameters::ACT_LIGHT_POSITION_VIEW_SPACE },
            { "lightDir",           GpuProgramParameters::ACT_LIGHT_DIRECTION_VIEW_SPACE },
            { "spotParams",         GpuProgramParameters::ACT_SPOTLIGHT_PARAMS },
            { "farClipDistance",    GpuProgramParameters::ACT_FAR_CLIP_DISTANCE },
            { "shadowViewProjMat",  GpuProgramParameters::ACT_TEXTURE_VIEWPROJ_MATRIX }
        };
        int numParams = sizeof(AUTO_PARAMS) / sizeof(AutoParamPair);
        
        for (int i=0; i<numParams; i++)
        {
            if (params->_findNamedConstantDefinition(AUTO_PARAMS[i].name))
            {
                params->setNamedAutoConstant(AUTO_PARAMS[i].name, AUTO_PARAMS[i].type);
            }
        }
    }
};

LightMaterialGenerator::LightMaterialGenerator()
{
    vsMask = 0x00000004;
    fsMask = 0x0000003F;
    matMask =   LightMaterialGenerator::MI_DIRECTIONAL | 
                LightMaterialGenerator::MI_SHADOW_CASTER;
    
    materialBaseName = "DeferredShading/LightMaterial/";
    if ((GpuProgramManager::getSingleton().isSyntaxSupported("glsl") || GpuProgramManager::getSingleton().isSyntaxSupported("glsles")) &&
        !(GpuProgramManager::getSingleton().isSyntaxSupported("ps_2_x") ||GpuProgramManager::getSingleton().isSyntaxSupported("arbfp1")))
        mImpl = new LightMaterialGeneratorGLSL("DeferredShading/LightMaterial/");
    else
        mImpl = new LightMaterialGeneratorCG("DeferredShading/LightMaterial/");
}

LightMaterialGenerator::~LightMaterialGenerator()
{

}
